/*
 * Decompiled with CFR 0.152.
 */
package com.cbmportal.portal.services.impl;

import com.auth0.jwt.JWT;
import com.auth0.jwt.interfaces.DecodedJWT;
import com.cbmportal.portal.domains.PortalUser;
import com.cbmportal.portal.domains.RefreshToken;
import com.cbmportal.portal.domains.UserPermission;
import com.cbmportal.portal.domains.UserRole;
import com.cbmportal.portal.domains.VO.AuthUser;
import com.cbmportal.portal.domains.VO.LoginResponseVO;
import com.cbmportal.portal.exceptions.AuthenticationFailedException;
import com.cbmportal.portal.exceptions.ValidationException;
import com.cbmportal.portal.repositories.PortalUserRepository;
import com.cbmportal.portal.repositories.UserRoleRepository;
import com.cbmportal.portal.services.AuthenticationService;
import com.cbmportal.portal.services.SecurityNotificationService;
import com.cbmportal.portal.services.TokenService;
import com.cbmportal.portal.services.impl.FailedLoginAttemptService;
import com.nimbusds.jose.JOSEException;
import java.text.ParseException;
import java.time.Instant;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

@Service
@Transactional
public class AuthenticationServiceImpl
implements AuthenticationService {
    Logger log = LoggerFactory.getLogger(AuthenticationServiceImpl.class);
    private final PortalUserRepository portalUserRepository;
    private final UserRoleRepository userRoleRepository;
    private final PasswordEncoder passwordEncoder;
    private final AuthenticationManager authenticationManager;
    private final TokenService tokenService;
    private final SecurityNotificationService securityNotificationService;
    private final FailedLoginAttemptService failedLoginAttemptService;

    public AuthenticationServiceImpl(PortalUserRepository portalUserRepository, UserRoleRepository userRoleRepository, PasswordEncoder passwordEncoder, AuthenticationManager authenticationManager, TokenService tokenService, SecurityNotificationService securityNotificationService, FailedLoginAttemptService failedLoginAttemptService) {
        this.portalUserRepository = portalUserRepository;
        this.userRoleRepository = userRoleRepository;
        this.passwordEncoder = passwordEncoder;
        this.authenticationManager = authenticationManager;
        this.tokenService = tokenService;
        this.securityNotificationService = securityNotificationService;
        this.failedLoginAttemptService = failedLoginAttemptService;
    }

    @Override
    public PortalUser registerUser(PortalUser portalUser) {
        boolean existsTrimmed;
        String correlationId = UUID.randomUUID().toString();
        String rawUsername = portalUser.getUsername();
        if (rawUsername == null || rawUsername.isBlank()) {
            this.log.warn("[cid:{}] Registration failed - empty username provided", (Object)correlationId);
            throw new ValidationException("Username is required");
        }
        this.log.debug("[cid:{}] Registration attempt - raw username received: '{}'", (Object)correlationId, (Object)rawUsername);
        String trimmed = rawUsername.trim();
        String normalized = trimmed.toLowerCase();
        this.log.debug("[cid:{}] Registration attempt - trimmed: '{}', normalized: '{}'", new Object[]{correlationId, trimmed, normalized});
        Optional<PortalUser> existingOpt = this.portalUserRepository.findPortalUserByUserNameIgnoreCase(normalized);
        boolean existsIgnoreCase = existingOpt.isPresent();
        boolean existsRaw = this.portalUserRepository.findPortalUserByUserName(rawUsername).isPresent();
        boolean bl = existsTrimmed = !rawUsername.equals(trimmed) && this.portalUserRepository.findPortalUserByUserName(trimmed).isPresent();
        if (existsIgnoreCase || existsRaw || existsTrimmed) {
            if (existingOpt.isPresent() && Boolean.TRUE.equals(existingOpt.get().getDisabled())) {
                PortalUser existing = existingOpt.get();
                this.log.info("[cid:{}] Reactivating previously disabled user '{}' (id {})", new Object[]{correlationId, existing.getUsername(), existing.getId()});
                if (portalUser.getPassword() == null || portalUser.getPassword().isBlank()) {
                    throw new ValidationException("Password is required for reactivation");
                }
                existing.setPassword(this.passwordEncoder.encode((CharSequence)portalUser.getPassword()));
                existing.setDisabled(false);
                existing.setUserFirst(portalUser.getUserFirst());
                existing.setUserLast(portalUser.getUserLast());
                existing.setEmail(portalUser.getEmail());
                UserPermission perm = portalUser.getPermission() == null ? UserPermission.DM : portalUser.getPermission();
                existing.setPermission(perm);
                if (existing.getAuthorities() == null) {
                    existing.setAuthorities(new HashSet<UserRole>());
                }
                this.userRoleRepository.findByAuthority(perm.name()).ifPresent(role -> {
                    if (existing.getAuthorities().stream().noneMatch(r -> r.getAuthority().equals(role.getAuthority()))) {
                        existing.getAuthorities().add(role);
                    }
                });
                try {
                    PortalUser saved = (PortalUser)this.portalUserRepository.save(existing);
                    this.log.info("[cid:{}] Reactivation success - user '{}' re-enabled", (Object)correlationId, (Object)saved.getUsername());
                    return saved;
                }
                catch (DataIntegrityViolationException dive) {
                    this.log.error("[cid:{}] Reactivation failed due to constraint violation for username '{}'", new Object[]{correlationId, normalized, dive});
                    throw new ValidationException("Username already exists");
                }
            }
            this.log.warn("[cid:{}] Registration blocked - username already exists (ignoreCase/raw/trimmed flags: {}/{}/{})", new Object[]{correlationId, existsIgnoreCase, existsRaw, existsTrimmed});
            throw new ValidationException("Username already exists");
        }
        if (portalUser.getPassword() == null || portalUser.getPassword().isBlank()) {
            this.log.warn("[cid:{}] Registration failed - empty password for username '{}'", (Object)correlationId, (Object)rawUsername);
            throw new ValidationException("Password is required");
        }
        String encodedPassword = this.passwordEncoder.encode((CharSequence)portalUser.getPassword());
        this.log.info("[cid:{}] Registering new user '{}' after uniqueness validation", (Object)correlationId, (Object)normalized);
        UserPermission permission = portalUser.getPermission() == null ? UserPermission.DM : portalUser.getPermission();
        UserRole primaryRole = this.userRoleRepository.findByAuthority(permission.name()).orElseThrow(() -> new IllegalStateException(permission.name() + " role not found"));
        HashSet<UserRole> authorities = new HashSet<UserRole>();
        authorities.add(primaryRole);
        portalUser.setUserName(normalized);
        portalUser.setPassword(encodedPassword);
        portalUser.setAuthorities(authorities);
        portalUser.setPermission(permission);
        try {
            PortalUser saved = (PortalUser)this.portalUserRepository.save(portalUser);
            this.log.info("[cid:{}] Registration success - user '{}' saved with id {}", new Object[]{correlationId, normalized, saved.getId()});
            return saved;
        }
        catch (DataIntegrityViolationException dive) {
            this.log.error("[cid:{}] Registration failed - constraint violation for username '{}'", new Object[]{correlationId, normalized, dive});
            throw new ValidationException("Username already exists");
        }
    }

    @Override
    public LoginResponseVO loginUser(String username, String password, boolean rememberMe, String deviceInfo, String ipAddress) {
        String normalizedUsername = username.toLowerCase();
        this.log.info("Authentication attempt for user: {}, rememberMe: {}", (Object)normalizedUsername, (Object)rememberMe);
        if (this.failedLoginAttemptService.isAccountLocked(normalizedUsername)) {
            long remainingSeconds = this.failedLoginAttemptService.getRemainingLockoutSeconds(normalizedUsername);
            long remainingMinutes = remainingSeconds / 60L;
            this.log.warn("Login attempt for locked account: {}, remaining lockout: {} minutes", (Object)normalizedUsername, (Object)remainingMinutes);
            throw new AuthenticationFailedException("Account is temporarily locked due to multiple failed login attempts. Please try again in " + remainingMinutes + " minutes.");
        }
        try {
            Authentication auth = this.authenticationManager.authenticate((Authentication)new UsernamePasswordAuthenticationToken((Object)normalizedUsername, (Object)password));
            String accessToken = this.tokenService.generateJwt(auth);
            RefreshToken refreshToken = this.tokenService.generateRefreshToken(auth, rememberMe, deviceInfo, ipAddress);
            LoginResponseVO responseVO = new LoginResponseVO();
            AuthUser recoveredUser = new AuthUser();
            PortalUser portalUser = this.portalUserRepository.findPortalUserByUserName(normalizedUsername).orElseThrow(() -> new UsernameNotFoundException("User not found: " + normalizedUsername));
            this.setRecoveredUsersValues(recoveredUser, portalUser);
            responseVO.setUser(recoveredUser);
            responseVO.setJwt(accessToken);
            responseVO.setRefreshToken(refreshToken.getToken());
            responseVO.setJwtStatus(true);
            this.log.info("Authentication successful for user: {} with role: {}, refresh token expires: {}", new Object[]{normalizedUsername, portalUser.getPermission(), refreshToken.getExpiresAt()});
            this.failedLoginAttemptService.resetFailedAttempts(normalizedUsername);
            try {
                this.securityNotificationService.notifyNewDeviceLogin(portalUser, refreshToken);
            }
            catch (Exception e) {
                this.log.error("Failed to send new device notification, continuing with login", (Throwable)e);
            }
            return responseVO;
        }
        catch (AuthenticationException authExc) {
            this.log.error("Authentication failed for user: {} - Invalid credentials", (Object)normalizedUsername);
            this.failedLoginAttemptService.recordFailedAttempt(normalizedUsername, ipAddress);
            if (this.failedLoginAttemptService.isAccountLocked(normalizedUsername)) {
                long remainingSeconds = this.failedLoginAttemptService.getRemainingLockoutSeconds(normalizedUsername);
                long remainingMinutes = remainingSeconds / 60L;
                throw new AuthenticationFailedException("Too many failed login attempts. Account is locked for " + remainingMinutes + " minutes.");
            }
            throw new AuthenticationFailedException("Invalid username or password", authExc);
        }
        catch (JOSEException | ParseException exc) {
            this.log.error("Token generation failed for user: {}", (Object)normalizedUsername, (Object)exc);
            throw new AuthenticationFailedException("Failed to generate authentication token", exc);
        }
    }

    @Override
    public LoginResponseVO refreshAccessToken(String refreshToken) {
        this.log.info("Refresh token request received");
        try {
            Authentication auth = this.tokenService.validateRefreshToken(refreshToken);
            String newAccessToken = this.tokenService.generateJwt(auth);
            PortalUser portalUser = this.portalUserRepository.findPortalUserByUserName(auth.getName()).orElseThrow(() -> new UsernameNotFoundException("User not found: " + auth.getName()));
            AuthUser recoveredUser = new AuthUser();
            this.setRecoveredUsersValues(recoveredUser, portalUser);
            LoginResponseVO responseVO = new LoginResponseVO();
            responseVO.setUser(recoveredUser);
            responseVO.setJwt(newAccessToken);
            responseVO.setRefreshToken(refreshToken);
            responseVO.setJwtStatus(true);
            this.log.info("Access token refreshed successfully for user: {}", (Object)auth.getName());
            return responseVO;
        }
        catch (IllegalArgumentException exc) {
            this.log.error("Refresh token validation failed: {}", (Object)exc.getMessage());
            throw new AuthenticationFailedException("Invalid or expired refresh token", exc);
        }
        catch (JOSEException | ParseException exc) {
            this.log.error("Token generation failed during refresh", exc);
            throw new AuthenticationFailedException("Failed to generate new access token", exc);
        }
    }

    @Override
    public void logout(String refreshToken) {
        this.log.info("Logout request received");
        try {
            this.tokenService.revokeRefreshToken(refreshToken);
            this.log.info("Logout successful, refresh token revoked");
        }
        catch (IllegalArgumentException exc) {
            this.log.warn("Logout attempted with invalid refresh token: {}", (Object)exc.getMessage());
        }
    }

    private void setRecoveredUsersValues(AuthUser recoveredUser, PortalUser portalUser) {
        recoveredUser.setId(portalUser.getId());
        recoveredUser.setAuthorities((Set<UserRole>)portalUser.getAuthorities());
        recoveredUser.setDistrict(portalUser.getDistrict());
        recoveredUser.setUserFirst(portalUser.getUserFirst());
        recoveredUser.setUserLast(portalUser.getUserLast());
        recoveredUser.setEmail(portalUser.getEmail());
        recoveredUser.setUserName(portalUser.getUsername());
        recoveredUser.setPermission(portalUser.getPermission());
    }

    @Override
    public LoginResponseVO validateJwt(String token) {
        try {
            DecodedJWT jwtDecoder = JWT.decode((String)token);
            if (jwtDecoder.getExpiresAtAsInstant().isAfter(Instant.now())) {
                LoginResponseVO authResponse = new LoginResponseVO();
                PortalUser portalUser = this.portalUserRepository.findPortalUserByUserName(jwtDecoder.getSubject()).orElseThrow(() -> new UsernameNotFoundException("User not found: " + jwtDecoder.getSubject()));
                AuthUser recoveredUser = new AuthUser();
                this.setRecoveredUsersValues(recoveredUser, portalUser);
                authResponse.setJwt(token);
                authResponse.setJwtStatus(true);
                authResponse.setUser(recoveredUser);
                return authResponse;
            }
            this.log.warn("Token expired for user: {}", (Object)jwtDecoder.getSubject());
            throw new AuthenticationFailedException("Token has expired");
        }
        catch (AuthenticationException authExc) {
            this.log.error("Token validation failed", (Throwable)authExc);
            throw new AuthenticationFailedException("Invalid token", authExc);
        }
        catch (Exception exc) {
            this.log.error("Unexpected error during token validation", (Throwable)exc);
            throw new AuthenticationFailedException("Token validation failed", exc);
        }
    }
}

